/*
 * Decompiled with CFR 0.152.
 */
package info.ata4.bspsrc.lib;

import info.ata4.bspsrc.common.util.PathUtil;
import info.ata4.bspsrc.lib.BspFileReader;
import info.ata4.bspsrc.lib.PakFile;
import info.ata4.bspsrc.lib.exceptions.BspException;
import info.ata4.bspsrc.lib.exceptions.GoldSrcFormatException;
import info.ata4.bspsrc.lib.exceptions.ZipFileBspException;
import info.ata4.bspsrc.lib.io.XORUtils;
import info.ata4.bspsrc.lib.lump.GameLump;
import info.ata4.bspsrc.lib.lump.Lump;
import info.ata4.bspsrc.lib.lump.LumpFile;
import info.ata4.bspsrc.lib.lump.LumpType;
import info.ata4.bspsrc.lib.util.StringMacroUtils;
import info.ata4.io.DataReader;
import info.ata4.io.DataReaders;
import info.ata4.io.DataWriter;
import info.ata4.io.DataWriters;
import info.ata4.io.Seekable;
import info.ata4.io.buffer.ByteBufferUtils;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BspFile {
    private static final Logger L = LogManager.getLogger();
    public static final int BSP_ID = StringMacroUtils.makeID("VBSP");
    public static final int BSP_ID_TF = StringMacroUtils.makeID("rBSP");
    private ByteOrder bo;
    public static final int HEADER_LUMPS = 64;
    public static final int HEADER_LUMPS_TF = 128;
    public static final int HEADER_SIZE = 1036;
    public static final int MAX_LUMPFILES = 128;
    private Path file;
    private String name;
    private final List<Lump> lumps = new ArrayList<Lump>(64);
    private final List<GameLump> gameLumps = new ArrayList<GameLump>();
    private int version;
    private int mapRev;
    private int appId = 0;

    public BspFile() {
    }

    public BspFile(Path file, boolean memMapping) throws BspException, IOException {
        this.loadImpl(file, memMapping);
    }

    public BspFile(Path file) throws BspException, IOException {
        this.loadImpl(file);
    }

    private void loadImpl(Path file) throws BspException, IOException {
        this.load(file);
    }

    private void loadImpl(Path file, boolean memMapping) throws BspException, IOException {
        this.load(file, memMapping);
    }

    public void load(Path file, boolean memMapping) throws BspException, IOException {
        this.file = file;
        this.name = PathUtil.nameWithoutExtension(file).orElse(null);
        L.debug("Loading headers from {}", (Object)this.name);
        ByteBuffer bb = this.createBuffer(memMapping);
        L.trace("Endianness: {}", (Object)this.bo);
        bb.order(this.bo);
        this.version = bb.getInt();
        L.trace("Version: {}", (Object)this.version);
        if (this.version == 262164) {
            L.trace("Found Dark Messiah header");
            this.appId = 2100;
            this.version &= 0xFF;
        } else if (this.version == 27) {
            L.trace("Found Contagion header");
            this.appId = 238430;
        }
        if (this.version == 21 && bb.getInt(8) == 0) {
            L.trace("Found Left 4 Dead 2 header");
            this.appId = 550;
        }
        if (this.appId == 238430) {
            bb.getInt();
        }
        if (this.appId == 1454890) {
            this.mapRev = bb.getInt();
            L.trace("Map revision: {}", (Object)this.mapRev);
            bb.getInt();
        }
        this.loadLumps(bb);
        this.loadGameLumps();
        if (this.appId == 1454890) {
            this.loadTitanfallLumpFiles();
            this.loadTitanfallEntityFiles();
        } else {
            this.mapRev = bb.getInt();
            L.trace("Map revision: {}", (Object)this.mapRev);
        }
    }

    public void load(Path file) throws BspException, IOException {
        this.load(file, true);
    }

    public void save(Path file) throws IOException {
        this.file = file;
        this.name = file.getFileName().toString();
        L.debug("Saving headers to {}", (Object)this.name);
        this.saveGameLumps();
        int size = this.fixLumpOffsets();
        MappedByteBuffer bb = ByteBufferUtils.openReadWrite(file, 0, size);
        bb.order(this.bo);
        bb.putInt(BSP_ID);
        bb.putInt(this.version);
        this.saveLumps(bb);
        bb.putInt(this.mapRev);
    }

    private ByteBuffer createBuffer(boolean memMapping) throws IOException, BspException {
        ByteBuffer bb = memMapping ? ByteBufferUtils.openReadOnly(this.file) : ByteBufferUtils.load(this.file);
        if (bb.capacity() < 4) {
            throw new BspException("Invalid or missing header");
        }
        int ident = bb.getInt();
        if (ident == 1347093252 || ident == 1347093766 || ident == 1347094280) {
            L.error("File is a zip archive. Make sure to first extract any bsp file it might contain and then select these for decompilation.");
            throw new ZipFileBspException("Loaded file is a zip archive");
        }
        if (bb.capacity() < 1036) {
            throw new BspException("Invalid or missing header");
        }
        if (ident == BSP_ID) {
            this.bo = ByteOrder.BIG_ENDIAN;
            return bb;
        }
        if ((ident = Integer.reverseBytes(ident)) == BSP_ID) {
            this.bo = ByteOrder.LITTLE_ENDIAN;
            return bb;
        }
        if (ident == BSP_ID_TF) {
            L.trace("Found Titanfall header");
            this.appId = 1454890;
            this.bo = ByteOrder.LITTLE_ENDIAN;
            return bb;
        }
        if (ident == 30) {
            throw new GoldSrcFormatException("The GoldSrc format is not supported");
        }
        byte[] mapKey = new byte[32];
        bb.position(384);
        bb.get(mapKey);
        int identXor = XORUtils.xor(ident, mapKey);
        if (identXor == BSP_ID) {
            this.bo = ByteOrder.LITTLE_ENDIAN;
            L.debug("Found Tactical Intervention XOR encryption using the key \"{}\"", (Object)new String(mapKey));
            if (memMapping || bb.isReadOnly()) {
                bb = ByteBufferUtils.load(this.file);
            }
            XORUtils.xor(bb, mapKey);
            bb.position(4);
            return bb;
        }
        throw new BspException("Unknown file ident: " + ident + " (" + StringMacroUtils.unmakeID(ident) + ")");
    }

    private void loadLumps(ByteBuffer bb) {
        L.debug("Loading lumps");
        int numLumps = this.appId == 1454890 ? 128 : 64;
        for (int i = 0; i < numLumps; ++i) {
            int len;
            int ofs;
            int vers;
            if (this.appId == 550) {
                vers = bb.getInt();
                ofs = bb.getInt();
                len = bb.getInt();
            } else {
                ofs = bb.getInt();
                len = bb.getInt();
                vers = bb.getInt();
            }
            int fourCC = bb.getInt();
            LumpType ltype = LumpType.get(i, this.version);
            if (ofs > bb.limit()) {
                int ofsOld = ofs;
                ofs = bb.limit();
                len = 0;
                L.warn("Invalid lump offset {} in {}, assuming {}", (Object)ofsOld, (Object)ltype, (Object)ofs);
            } else if (ofs < 0) {
                int ofsOld = ofs;
                ofs = 0;
                len = 0;
                L.warn("Negative lump offset {} in {}, assuming {}", (Object)ofsOld, (Object)ltype, (Object)ofs);
            }
            if (ofs + len > bb.limit()) {
                lenOld = len;
                len = bb.limit() - ofs;
                L.warn("Invalid lump length {} in {}, assuming {}", (Object)lenOld, (Object)ltype, (Object)len);
            } else if (len < 0) {
                lenOld = len;
                len = 0;
                L.warn("Negative lump length {} in {}, assuming {}", (Object)lenOld, (Object)ltype, (Object)len);
            }
            Lump l = new Lump(i, ltype);
            l.setBuffer(ByteBufferUtils.getSlice(bb, ofs, len));
            l.setOffset(ofs);
            l.setParentFile(this.file);
            l.setFourCC(fourCC);
            l.setVersion(vers);
            this.lumps.add(l);
        }
    }

    private void saveLumps(ByteBuffer bb) {
        L.debug("Saving lumps");
        for (Lump lump : this.lumps) {
            if (this.appId == 550) {
                bb.putInt(lump.getVersion());
                bb.putInt(lump.getOffset());
                bb.putInt(lump.getLength());
            } else {
                bb.putInt(lump.getOffset());
                bb.putInt(lump.getLength());
                bb.putInt(lump.getVersion());
            }
            bb.putInt(lump.getFourCC());
            if (lump.getLength() == 0) continue;
            if (lump.getType() == LumpType.LUMP_GAME_LUMP) {
                this.fixGameLumpOffsets(lump);
            }
            ByteBuffer lbb = lump.getBuffer();
            bb.mark();
            bb.position(lump.getOffset());
            bb.put(lbb);
            bb.reset();
        }
    }

    public void loadLumpFiles() {
        Path lumpFile;
        L.debug("Loading lump files");
        for (int i = 0; i < 128 && Files.exists(lumpFile = this.file.resolveSibling(String.format("%s_l_%d.lmp", this.name, i)), new LinkOption[0]); ++i) {
            try {
                LumpFile lumpFileExt = new LumpFile(this.version);
                lumpFileExt.load(lumpFile, this.bo);
                Lump l = lumpFileExt.getLump();
                this.lumps.set(l.getIndex(), l);
                if (l.getType() != LumpType.LUMP_GAME_LUMP) continue;
                this.gameLumps.clear();
                this.loadGameLumps();
                continue;
            }
            catch (IOException ex) {
                L.warn("Unable to load lump file " + String.valueOf(lumpFile.getFileName()), (Throwable)ex);
            }
        }
    }

    private void loadTitanfallLumpFiles() {
        L.debug("Loading Titanfall lump files");
        for (int i = 0; i < 128; ++i) {
            Path lumpFile = this.file.resolveSibling(String.format("%s.bsp.%04x.bsp_lump", this.name, i));
            if (!Files.exists(lumpFile, new LinkOption[0])) continue;
            Lump l = this.lumps.get(i);
            try {
                MappedByteBuffer bb = ByteBufferUtils.openReadOnly(lumpFile);
                bb.order(this.bo);
                l.setBuffer(bb);
                l.setParentFile(lumpFile);
                continue;
            }
            catch (IOException ex) {
                L.warn("Unable to load lump file " + String.valueOf(lumpFile.getFileName()), (Throwable)ex);
            }
        }
    }

    private void loadTitanfallEntityFiles() {
        L.debug("Loading Titanfall entity files");
        Lump entlump = this.getLump(LumpType.LUMP_ENTITIES);
        ByteBuffer bbEnt = entlump.getBuffer();
        bbEnt.limit(bbEnt.limit() - 1);
        ArrayList<ByteBuffer> bbList = new ArrayList<ByteBuffer>();
        bbList.add(bbEnt);
        bbList.add(this.loadTitanfallEntityFile("env"));
        bbList.add(this.loadTitanfallEntityFile("fx"));
        bbList.add(this.loadTitanfallEntityFile("script"));
        bbList.add(this.loadTitanfallEntityFile("snd"));
        bbList.add(this.loadTitanfallEntityFile("spawn"));
        bbList.add(ByteBuffer.wrap(new byte[]{0}));
        ByteBuffer bbEntNew = ByteBufferUtils.concat(bbList);
        entlump.setBuffer(bbEntNew);
    }

    private ByteBuffer loadTitanfallEntityFile(String entname) {
        Path entFile = this.file.resolveSibling(String.format("%s_%s.ent", this.name, entname));
        ByteBuffer bb = ByteBuffer.allocate(0);
        try {
            if (Files.exists(entFile, new LinkOption[0]) && Files.size(entFile) > 12L) {
                bb = ByteBufferUtils.load(entFile);
                bb.position(11);
                bb.limit(bb.capacity() - 1);
                bb = bb.slice();
            }
        }
        catch (IOException ex) {
            L.warn("Unable to load entity file " + String.valueOf(entFile.getFileName()), (Throwable)ex);
        }
        return bb;
    }

    private void loadGameLumps() {
        L.debug("Loading game lumps");
        try {
            Lump lump = this.getLump(LumpType.LUMP_GAME_LUMP);
            DataReader in = DataReaders.forByteBuffer(lump.getBuffer());
            if (this.version == 20 && this.bo == ByteOrder.LITTLE_ENDIAN && this.checkInvalidHeaders(in, false) && !this.checkInvalidHeaders(in, true)) {
                L.trace("Found Vindictus game lump header");
                this.appId = 212160;
            }
            int glumps = in.readInt();
            for (int i = 0; i < glumps; ++i) {
                String glName;
                int vers;
                int flags;
                if (this.appId == 2100) {
                    in.readInt();
                }
                int fourCC = in.readInt();
                if (this.appId == 212160) {
                    flags = in.readInt();
                    vers = in.readInt();
                } else {
                    flags = in.readUnsignedShort();
                    vers = in.readUnsignedShort();
                }
                int ofs = in.readInt();
                int len = in.readInt();
                if (flags == 1) {
                    in.seek(8L, Seekable.Origin.CURRENT);
                    int nextOfs = in.readInt();
                    if (nextOfs == 0) {
                        nextOfs = lump.getOffset() + lump.getLength();
                    }
                    len = nextOfs - ofs;
                    in.seek(-12L, Seekable.Origin.CURRENT);
                }
                if (ofs - lump.getOffset() > 0) {
                    ofs -= lump.getOffset();
                }
                if ((glName = StringMacroUtils.unmakeID(fourCC)).trim().isEmpty()) {
                    glName = "<dummy>";
                }
                if (ofs > lump.getLength()) {
                    int ofsOld = ofs;
                    ofs = lump.getLength();
                    len = 0;
                    L.warn("Invalid game lump offset {} in {}, assuming {}", (Object)ofsOld, (Object)glName, (Object)ofs);
                } else if (ofs < 0) {
                    int ofsOld = ofs;
                    ofs = 0;
                    len = 0;
                    L.warn("Negative game lump offset {} in {}, assuming {}", (Object)ofsOld, (Object)glName, (Object)ofs);
                }
                if (ofs + len > lump.getLength()) {
                    lenOld = len;
                    len = lump.getLength() - ofs;
                    L.warn("Invalid game lump length {} in {}, assuming {}", (Object)lenOld, (Object)glName, (Object)len);
                } else if (len < 0) {
                    lenOld = len;
                    len = 0;
                    L.warn("Negative game lump length {} in {}, assuming {}", (Object)lenOld, (Object)glName, (Object)len);
                }
                GameLump gl = new GameLump();
                gl.setBuffer(ByteBufferUtils.getSlice(lump.getBuffer(), ofs, len));
                gl.setOffset(ofs);
                gl.setFourCC(fourCC);
                gl.setFlags(flags);
                gl.setVersion(vers);
                this.gameLumps.add(gl);
            }
            L.debug("Game lumps: {}", (Object)glumps);
        }
        catch (IOException ex) {
            L.error("Couldn't load game lumps", (Throwable)ex);
        }
    }

    private void saveGameLumps() {
        L.debug("Saving game lumps");
        int headerSize = 4;
        headerSize = this.appId == 212160 ? (headerSize += 20 * this.gameLumps.size()) : (headerSize += 16 * this.gameLumps.size());
        int dataSize = 0;
        for (GameLump gl : this.gameLumps) {
            dataSize += gl.getLength();
        }
        try {
            ByteBuffer bb = ByteBuffer.allocateDirect(headerSize + dataSize);
            bb.order(this.bo);
            DataWriter out = DataWriters.forByteBuffer(bb);
            out.writeInt(this.gameLumps.size());
            int offset = headerSize;
            for (GameLump gl : this.gameLumps) {
                gl.setOffset(offset);
                offset += gl.getLength();
                out.writeInt(gl.getFourCC());
                if (this.appId == 212160) {
                    out.writeInt(gl.getFlags());
                    out.writeInt(gl.getVersion());
                } else {
                    out.writeUnsignedShort(gl.getFlags());
                    out.writeUnsignedShort(gl.getVersion());
                }
                out.writeInt(gl.getOffset());
                out.writeInt(gl.getLength());
                bb.mark();
                bb.position(gl.getOffset());
                bb.put(gl.getBuffer());
                bb.reset();
            }
            Lump gameLump = this.getLump(LumpType.LUMP_GAME_LUMP);
            gameLump.setBuffer(bb);
        }
        catch (IOException ex) {
            L.error("Couldn''t save game lumps", (Throwable)ex);
        }
    }

    private int fixLumpOffsets() {
        int offset = 1036;
        for (Lump lump : this.lumps) {
            if (lump.getLength() == 0) {
                lump.setOffset(0);
                continue;
            }
            lump.setOffset(offset);
            offset += lump.getLength();
        }
        return offset;
    }

    private void fixGameLumpOffsets(Lump lump) {
        ByteBuffer bb = lump.getBuffer();
        int glumps = bb.getInt();
        for (int i = 0; i < glumps; ++i) {
            int index = this.appId == 212160 ? 20 * i + 16 : 16 * i + 12;
            int ofs = bb.getInt(index);
            bb.putInt(index, ofs += lump.getOffset());
        }
    }

    private boolean checkInvalidHeaders(DataReader in, boolean vin) throws IOException {
        int glumps = in.readInt();
        for (int i = 0; i < glumps; ++i) {
            String glName = StringMacroUtils.unmakeID(in.readInt());
            if (!glName.matches("^[a-zA-Z0-9]{4}$")) {
                in.position(0L);
                return true;
            }
            in.seek(vin ? 16L : 12L, Seekable.Origin.CURRENT);
        }
        in.position(0L);
        return false;
    }

    public List<Lump> getLumps() {
        return Collections.unmodifiableList(this.lumps);
    }

    public Lump getLump(LumpType type) {
        return this.lumps.get(type.getIndex());
    }

    public List<GameLump> getGameLumps() {
        return Collections.unmodifiableList(this.gameLumps);
    }

    public GameLump getGameLump(String sid) {
        for (GameLump gl : this.gameLumps) {
            if (!gl.getName().equalsIgnoreCase(sid)) continue;
            return gl;
        }
        return null;
    }

    public void compress() {
        L.info("Compressing lumps");
        for (Lump l : this.lumps) {
            if (l.getType() == LumpType.LUMP_GAME_LUMP || l.getType() == LumpType.LUMP_PAKFILE || l.getLength() <= 17 || l.isCompressed()) continue;
            L.debug("Compressing {}", (Object)l.getName());
            l.compress();
        }
        for (GameLump gl : this.gameLumps) {
            if (gl.getLength() <= 17 || gl.isCompressed()) continue;
            L.debug("Compressing {}", (Object)gl.getName());
            gl.compress();
        }
        this.gameLumps.add(new GameLump());
    }

    public void uncompress() {
        L.info("Uncompressing lumps");
        for (Lump l : this.lumps) {
            if (!l.isCompressed()) continue;
            l.uncompress();
        }
        for (GameLump gl : this.gameLumps) {
            if (!gl.isCompressed()) continue;
            gl.uncompress();
        }
        if (!this.gameLumps.isEmpty() && this.gameLumps.get(this.gameLumps.size() - 1).getLength() == 0) {
            this.gameLumps.remove(this.gameLumps.size() - 1);
        }
    }

    public boolean isCompressed() {
        for (Lump l : this.lumps) {
            if (!l.isCompressed()) continue;
            return true;
        }
        return false;
    }

    public PakFile getPakFile() {
        return new PakFile(this);
    }

    public boolean canReadLump(LumpType type) {
        return type.getBspVersion() == -1 || type.getBspVersion() <= this.version;
    }

    public Path getNextLumpFile() {
        for (int i = 0; i < 128; ++i) {
            Path lumpFile = this.file.resolveSibling(String.format("%s_l_%d.lmp", this.name, i));
            if (Files.exists(lumpFile, new LinkOption[0])) continue;
            return lumpFile;
        }
        return null;
    }

    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Path getFile() {
        return this.file;
    }

    public int getVersion() {
        return this.version;
    }

    public void setVersion(int version) throws BspException {
        this.version = version;
    }

    public int getRevision() {
        return this.mapRev;
    }

    public void setRevision(int mapRev) {
        this.mapRev = mapRev;
    }

    public ByteOrder getByteOrder() {
        return this.bo;
    }

    public int getAppId() {
        return this.appId;
    }

    public void setAppId(int appId) {
        this.appId = appId;
    }

    public BspFileReader getReader() {
        return new BspFileReader(this);
    }
}

